*This program performs the US Census Bureau's X12-ARIMA default procedure to seasonally adjust a data series. It works only when running STATA in windows.

program define x12
	version 12
	syntax anything [, data_options(string) x12_options(string) x11_options(string)]
	tokenize `anything'
	local year = "`1'"
	local date = "`2'"
	local variable = "`3'"
	local period = "`4'"
	
	disp "Seasonal Adjustment Program"
	disp "year variable name = `year'"
	disp "month or quarterly variable name = `date'"
	disp "variable to be seasonally adjusted = `variable'"
	disp "number of periods in a year = `period'"
	
	
	
	
	***********************
	* Housekeeping
	***********************
	*Specify names for some temporary files
	local file = "x12datfile"
	local specfile = "x12specfile"
	tempfile tempsa
	local outfile = "x12outfile"
	tempfile tempdata
	
	*specify directory of the x12 executable
	local directory "..\programs\X12_ARIMA\x12a"
	
	*specify directory for temporary files to be stored
	local tempfolder "..\programs\X12_ARIMA\temp_files"
	
	
	
	
	*****************************************************
	* Create ".spc" file for x12 executable to read in
	*****************************************************
	
	*Outsheet the variables into a dat-file.
	disp "Outsheeting the variable... "
	outsheet `year' `date' `variable' using "`tempfolder'/`file'.dat" , replace noname nolabel
	
	*create the ".spc" file that the x12 executable will read, and write in the header
	file open spec_file using "`tempfolder'/`specfile'.spc" , write text replace
	file write spec_file "series{" _newline
	file write spec_file "format=datevalue" _newline
	file write spec_file "period=`period'"  _newline

	* Insert data file into the ".spc" file
	file write spec_file "file=" _char(34) "`tempfolder'/`file'.dat" _char(34) _newline

	* Insert any other options to the series() step here
	file write spec_file "`data_options'" _newline
	file write spec_file "}" _newline

	* Insert any other options to the overall x12 here
	file write spec_file "`x12_options'" _newline
	file write spec_file "x11{save=(d11) `x11_options'}"
	file close spec_file
	
	
	
	
	
	****************************************************************
	* Run the x12 executable on the spc file that was just created
	****************************************************************
	pwd
	shell "`directory'\x12a" "`tempfolder'/`specfile'" -o "`tempfolder'/`outfile'"
	disp "---------------- X12 Reported Errors:----------------"
	type "`tempfolder'/`outfile'.err"
	
	save `tempdata'
	clear
	
	
	
	****************************************
	* read in the seasonally adjusted data
	***********************************
	insheet using "`tempfolder'/`outfile'.d11" , names
	*drop first line
	drop in 1/1
	
	* destring the text-based SA data.
	destring temp_filesx12oud11, gen(`variable'_sa)
	destring date, replace
	gen `year' = floor(date/100)
	gen `date' = mod(date, 100)
	keep `year' `date' `variable'_sa
	sort `year' `date'
	save `tempsa' , replace
	
	
	*************************************************************
	* Merge in the seasonally adjusted data with the original data
	*************************************************************
	use `tempdata'
	sort `year' `date'
	merge 1:1 `year' `date' using "`tempsa'"
	assert _merge == 3
	drop _merge
	label variable `variable'_sa "`variable', SA"
	
	*erase temporary files"
	cap erase "`tempfolder'/x12datfile.dat"
	cap erase "`tempfolder'/x12specfile.spc"
	cap erase "`tempfolder'/x12outfile.err"
	cap erase "`tempfolder'/x12outfile.log"
	cap erase "`tempfolder'/x12outfile.out"
	cap erase "`tempfolder'/x12outfile.d11"
	
end x12
